import torch
import torch.nn as nn
import torch.nn.functional as F
from causally.model.utils import BayesianLinear
from causally.model.abstract_model import AbstractModel
from causally.model.utils import WassDistance

class UITE_WASS(AbstractModel):
    def __init__(self, config,dataset):
        super(UITE_WASS, self).__init__(config,dataset)
        self.in_feature = self.dataset.size[1]
        self.alpha = self.config['alpha']
        self.beta = self.config['beta']
        self.bn = self.config['bn']
        self.repre_layer_sizes =  [self.in_feature]+self.config['repre_layer_sizes']
        self.pred_layer_sizes = [self.config['repre_layer_sizes'][-1]]+self.config['pred_layer_sizes'] + [1]
        self.batch_size = self.config['train_batch_size']


        self.repre_layers = nn.Sequential(*[
            BayesianLinear(in_s,out_s) for in_s,out_s in zip(self.repre_layer_sizes[:-1],
                                                             self.repre_layer_sizes[1:])
        ])

        self.pred_layers_treated = nn.Sequential(*[
            BayesianLinear(in_s, out_s) for in_s, out_s in zip(self.pred_layer_sizes[:-1],
                                                               self.pred_layer_sizes[1:])
        ])

        # self.pred_layers_treated.add_module('out1',BayesianLinear(self.pred_layer_sizes[-1],1))


        self.pred_layers_control = nn.Sequential(*[
            BayesianLinear(in_s, out_s) for in_s, out_s in zip(self.pred_layer_sizes[:-1],
                                                               self.pred_layer_sizes[1:])
        ])
        # self.pred_layers_control.add_module('out0', BayesianLinear(self.pred_layer_sizes[-1],1))

        self.mse_loss = nn.MSELoss(reduction='none')
        self.wass_distance = WassDistance(eps=0.01,max_iter=30,device=self.device)

    def forward(self, x, t):

        self.repre = self.repre_layers(x)
        outputs = torch.where(t == 1, self.pred_layers_treated(self.repre),
                                 self.pred_layers_control(self.repre))
        log_prior = self.log_prior()
        log_variational_posterior = self.log_variational_posterior()
        if self.training:
            return outputs,log_variational_posterior,log_prior
        else:
            return outputs

    def get_repre(self, x, device):
        self.eval()
        with torch.no_grad():
            return self.repre_layers.to(device)(x.to(device))

    def calculate_loss(self, x,t,y,w):

        outputs, log_variational_posterior, log_prior = self.forward(x,t)
        negative_log_likelihood = torch.sum(self.mse_loss(outputs, y) * w)

        vlb_loss =(log_variational_posterior-log_prior) / self.batch_size + negative_log_likelihood
        imb_loss = self.wass_distance(self.repre, t, w.squeeze())
        loss = vlb_loss + self.alpha * imb_loss
        return loss

    def log_prior(self):
        expect_log_prior = 0
        for module in self.repre_layers:
            expect_log_prior += module.log_prior
        for module in self.pred_layers_treated:
            expect_log_prior += module.log_prior
        for module in self.pred_layers_control:
            expect_log_prior += module.log_prior

        return expect_log_prior


    def log_variational_posterior(self):

        expect_log_posterior = 0
        for module in self.repre_layers:
            expect_log_posterior += module.log_variational_posterior
        for module in self.pred_layers_treated:
            expect_log_posterior += module.log_variational_posterior
        for module in self.pred_layers_control:
            expect_log_posterior += module.log_variational_posterior

        return expect_log_posterior